<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\ProductDetailsModel;
use App\Models\ProductModel;
use App\Models\ProductPaymentModel;
use App\Models\PurchaseModel;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;

class MobPurchaseController extends Controller
{
    public function store(Request $request)
    {
        // 🔥 FIXED: Updated validation rules
        $validator = Validator::make($request->all(), [
            'branch_id' => 'required|integer',
            'supplier_id' => 'required|integer|exists:supplier,id',
            'purchase_date' => 'required|date',
            'products' => 'required|array|min:1',
            'products.*.product_id' => 'required|integer|exists:products,id',
            'products.*.qty' => 'required|integer|min:1',
            'products.*.unit_price' => 'required|numeric|min:0',
            'products.*.margin_price' => 'required|numeric|min:0',
            'paid_amount' => 'nullable|numeric|min:0',
            'payment_type' => 'required_with:paid_amount|nullable|in:cash,upi,online,card',
            'payment_date' => 'nullable|date',
        ]);

        if ($validator->fails()) {
            Log::error('Purchase validation failed', [
                'errors' => $validator->errors()->toArray(),
                'data' => $request->all()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            DB::beginTransaction();

            $branchId = $request->branch_id;
            $totalAmount = 0;

            // Calculate total amount
            foreach ($request->products as $productData) {
                $totalAmount += $productData['margin_price'] * $productData['qty'];
            }

            // Generate purchase code
            $purchaseCode = $this->generatePurchaseCode();

            $paymentAmount = $request->paid_amount ?? 0;
            $dueAmount = $totalAmount - $paymentAmount;

            $status = 'pending';
            if ($paymentAmount >= $totalAmount) {
                $status = 'paid';
            } elseif ($paymentAmount > 0) {
                $status = 'partial';
            }

            // 🔥 FIXED: Create purchase with correct data
            $purchase = PurchaseModel::create([
                'purchase_code' => $purchaseCode,
                'branch_id' => $branchId,
                'purchase_date' => $request->purchase_date,
                'total_amount' => $totalAmount,
                'paid_amount' => $paymentAmount, 
                'due_amount' => $dueAmount,
                'status' => $status,
                'notes' => $request->notes,
                'active_flag' => 'A'
            ]);

            Log::info('Purchase created', [
                'purchase_id' => $purchase->id,
                'total_amount' => $totalAmount,
                'paid_amount' => $paymentAmount,
                'due_amount' => $dueAmount,
                'status' => $status
            ]);

            // Create product details for each product
            foreach ($request->products as $productData) {
                $product = ProductModel::findOrFail($productData['product_id']);
                
                // Generate product details code
                $detailsCode = $this->generateProductDetailsCode();

                // Create product detail linked to purchase
                ProductDetailsModel::create([
                    'product_details_code' => $detailsCode,
                    'product_id' => $productData['product_id'],
                    'branch_id' => $branchId,
                    'name' => $product->name,
                    'native_name' => $product->native_name,
                    'code' => $product->code,
                    'category_id' => $product->category_id,
                    'supplier_id' => $request->supplier_id,
                    'unit_price' => $productData['unit_price'],
                    'margin_price' => $productData['margin_price'],
                    'qty' => $productData['qty'],
                    'status' => $status == 'paid' ? 2 : ($status == 'partial' ? 3 : 1),
                    'total_amount' => $productData['margin_price'] * $productData['qty'],
                    'description' => $productData['description'] ?? null,
                    'purchase_id' => $purchase->id,
                    'active_flag' => 'A'
                ]);

                // Update product stock and prices
                $product->update([
                    'qty' => ($product->qty ?? 0) + $productData['qty'],
                    'total_qty' => ($product->total_qty ?? 0) + $productData['qty'],
                    'unit_price' => $productData['unit_price'],
                    'margin_price' => $productData['margin_price'],
                    'supplier_id' => $request->supplier_id,
                ]);
            }

            // 🔥 FIXED: Create payment only if amount and type provided
            if ($paymentAmount > 0 && !empty($request->payment_type)) {
                ProductPaymentModel::create([
                    'product_dtl_id' => null,
                    'purchase_id' => $purchase->id,
                    'payment_type' => $request->payment_type,
                    'payment_date' => $request->payment_date ?? $request->purchase_date,
                    'total_amount' => $totalAmount,
                    'paid_amount' => $paymentAmount,
                    'active_flag' => 'A',
                    'type' => 'purchase_payment'
                ]);

                Log::info('Payment record created', [
                    'purchase_id' => $purchase->id,
                    'payment_type' => $request->payment_type,
                    'paid_amount' => $paymentAmount
                ]);
            }

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Purchase created successfully',
                'data' => [
                    'purchase_id' => $purchase->id,
                    'purchase_code' => $purchase->purchase_code,
                    'total_amount' => $totalAmount,
                    'paid_amount' => $paymentAmount,
                    'due_amount' => $dueAmount,
                    'status' => $status
                ]
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error creating purchase', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'data' => $request->all()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to create purchase',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // 🔥 FIXED: Updated getProductsForPurchase method
    public function getProductsForPurchase(Request $request)
    {
        try {
            $branchId = $request->branch_id;
            
            $products = ProductModel::with(['category', 'supplier']) 
                ->where('branch_id', $branchId)
                ->where('active_flag', 'A')
                ->orderBy('name')
                ->get();

            $processedData = $products->map(function($product) {
                return [
                    'id' => $product->id,
                    'name' => $product->name,
                    'native_name' => $product->native_name,
                    'code' => $product->code,
                    'hsn_code' => $product->hsn_code,
                    'category_id' => $product->category_id,
                    'category_name' => $product->category->name ?? null, 
                    'supplier_id' => $product->supplier_id,
                    'supplier_name' => $product->supplier->supplier_name ?? null, 
                    'unit_price' => $product->unit_price ?? 0.0,
                    'margin_price' => $product->margin_price ?? 0.0,
                    'current_qty' => $product->qty ?? 0,
                    'description' => $product->description,
                    'products_image' => $product->products_image,
                ];
            });

            return response()->json([
                'success' => true,
                'data' => $processedData
            ]);

        } catch (\Exception $e) {
            Log::error('Error fetching products for purchase: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch products for purchase',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // Keep other methods as they were...
    private function generatePurchaseCode(): string
    {
        $lastPurchase = PurchaseModel::orderBy('id', 'desc')->first();

        if ($lastPurchase && $lastPurchase->purchase_code) {
            $lastCode = $lastPurchase->purchase_code;
            $parts = explode('-', $lastCode);
            $prefix = $parts[0];
            $number = isset($parts[1]) ? (int) $parts[1] : 0;
            $newNumber = str_pad($number + 1, 3, '0', STR_PAD_LEFT);
            return $prefix . '-' . $newNumber;
        }

        return 'PUR-001';
    }

    private function generateProductDetailsCode(): string
    {
        $lastDetail = ProductDetailsModel::orderBy('product_details_sid', 'desc')->first();

        if ($lastDetail && $lastDetail->product_details_code) {
            $lastCode = $lastDetail->product_details_code;
            $parts = explode('-', $lastCode);
            $prefix = $parts[0];
            $number = isset($parts[1]) ? (int) $parts[1] : 0;
            $newNumber = str_pad($number + 1, 3, '0', STR_PAD_LEFT);
            return $prefix . '-' . $newNumber;
        }

        return 'PROD-001';
    }
}
